/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xpciapi.c
 * static and board related functions
 * -----------------------------------------------------------------*/

#include <xtypedef.h>

#include <xiocommo.h>
#include <xpciapi.h>
#include <xregcons.h>
#include <xregxdir.h>
#include <xregx10.h>
#include <xregx11.h>
#include <xsession.h>
#include <xversion.h>

#include <xaddrmap.h>
#include <xloader.h>

#include <xpci.h>
#include <timeout.h>

#include <xdownld.h>
#include <xpatlex.h>
#include <xutil.h>

#define RETRY_COUNT_AFTER_BOOT  300

/* Address of PCI-Register (to read out PME-status) */
#define BX_PCI_REG      0x00500002

static bx_errtype TestBestXDirectRegRead(
  bx_handletype handle,
  bx_int32 dir_addr,                                
  bx_int32 regsize,                                
  bx_int32 *reg_value
);

static bx_errtype BestXBoardResetAdv(bx_handletype handle,bx_int32 boot);

/*   default for        block size, fast host threshold */
#define BX_DEF_CAPIPROP  { 0x1F000UL, 0 }
#define FOUR_CP_INITS BX_DEF_CAPIPROP, BX_DEF_CAPIPROP,   \
      BX_DEF_CAPIPROP, BX_DEF_CAPIPROP

bx_int32 bestx_capi_prop [BX_MAXHANDLES + 1][BX_CAPIPROP_SIZE] = {
  FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS,
  FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS, FOUR_CP_INITS,
  BX_DEF_CAPIPROP
};

/* -----------------------------------------------------------------
 * Power Up Loader Functions
 * ----------------------------------------------------------------- */

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXPUProg
 *
 * Purpose: Stores the current settings as power-up settings by
 *          programming the PU-loader code into flash memory.
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXPUProg(bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXPUProg [puprog]");
  BX_TRY_VARS;

  /* PU loader is started, when:
  1. The PU DIP is set (must always be the case) AND
  2. Powering OR
  3. PCIRST# asserted (negative edge triggered) while DBI_CMD_REG=1 OR
  4. DBI register access to 0x401048 OR
  5. PCIRST# deasserted (positive edge triggered) (new Faust image may need to be loaded) (Faust FPGA only) OR
  6. Faust image has been reloaded 
  */
  
  /* Flash sectors for PULoader and PUPointer */
  static const bx_int32 s_puloaderSectors[] =
  {
    BX_LOADER_IMAGE, /* the puloader is here */
    0x3ffffc         /* the pointer is here */
  };

  bx_int8 PointerCode[]={
                          (bx_int8) (BX_LOADER_IMAGE>>24 &0xff),
                          (bx_int8) (BX_LOADER_IMAGE>>16 &0xff),
                          (bx_int8) (BX_LOADER_IMAGE>>8  &0xff),
                          (bx_int8) (BX_LOADER_IMAGE>>0  &0xff)
                        };

  bx_int8  LoaderCode[BX_LOADER_CODELENGTH]; /* 64KB loader code maximum */
  bx_int32 LoaderLength=0; /* current length of LoaderCode in bytes */

  BX_TRY_BEGIN
  {
    BX_TRY(BestXPULoaderCodeGet(handle,LoaderCode,&LoaderLength));  

    /* Erase flash for PULoader and PUPointer */
    BX_TRY(BestXEraseFlash (handle, sizeof(s_puloaderSectors)/sizeof(bx_int32),(bx_int32ptr) s_puloaderSectors));
    /* Write flash for PULoader */
    BX_TRY(BestXWriteFlash (handle, s_puloaderSectors[0], LoaderCode, LoaderLength));
    /* Write flash for PUPointer */
    BX_TRY(BestXWriteFlash (handle, s_puloaderSectors[1], PointerCode, 4));
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXPULoaderCodeGet
 *
 * Purpose: Assembles the PULoader Code
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXPULoaderCodeGet(bx_handletype handle, bx_int8ptr LoaderCode,bx_int32ptr LoaderLengthPtr)
{
#define SYNCTIME 50 /* number of NOPs to wait for syncing */
  
  BX_DECLARE_FUNCNAME("BestXPULoaderCodeGet []");
  BX_TRY_VARS;

  bx_int32 LoaderLength=0;

  bx_int32 mask;           /* Observer mask */
  bx_int32 val=0;          /* Helper */

  bx_int32  DbgCntr=0;   /* Debug incrementor counter */

  BX_TRY_BEGIN
  {
    /* For debugging:
       We use the resource lock register as 
       an incrementor counter during the execution of the PU loader.
       Remark: Resource lock register is set to zero by PwrOnRst signal,
               but not by PCIRST# !
    */

    /* Switch Mephisto LED on to indicate PU loader has started */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_SYS_CTRL_REG,0x1,LoaderLength)

    /* Init incrementor counter to one, indicating that PU loader has started */
    DbgCntr++; 
    BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
    BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)

    /*************************************************************************
      Restart PU loader at negative edge of PCIRST#
     *************************************************************************/

    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_DBI_CMD_REG,0x1,LoaderLength)

    /* Init incrementor counter */
    DbgCntr++; 
    BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
    BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)

    if (BestXHasFaustFPGA(handle))
    {
      /* 1. Check, wether we need to initiate a reload of a Faust image (may happen after pos. edge of PCIRST#)
         2. If yes:
            - Copy the PCI reset pattern from Faust to Coco.
            - Copy the image number from Faust to Coco.
            - Initiate the reload by writing Coco register.
            - Exit. (PU loader will be called again after image reloading is done).

         3. Check, wether an image reload has taken place.
         4. If yes:
            - Copy new PCI reset pattern from Coco to new Faust image.
      */

      /* Check, wether we need to initiate a reload of a Faust image (may happen after pos. edge of PCIRST#)
         Bit 4=1 => Reload; Bit 4=0 => do not reload */
      BX_LOADERCMD_READWORD(LoaderCode+LoaderLength,BX_REG_SYS_RESETPATT_F,LoaderLength)
      BX_LOADERCMD_COMPAREWORD(LoaderCode+LoaderLength,0x0010,0x0010,LoaderLength) /* Check bit 4 */

      /* Jump in case of no reload needed; continue in case of reload needed */
      BX_LOADERCMD_BRANCHNOTEQUAL(LoaderCode+LoaderLength,BX_LOADER_IMAGE+LoaderLength+(4+30+6+2),LoaderLength)
      /* 4 bytes for BRANCHNOTEQUAL, 30 bytes for Coco/Faust copying, 6 bytes for WRITEWORD, 2 bytes for END */

      /* If reload needed:   Copy the PCI reset pattern (incl. the reload bit at bitpos 4) from Faust to Coco */    
      BX_LOADERCMD_LOADA0(LoaderCode+LoaderLength,BX_REG_SYS_RESETPATT_F,LoaderLength) /* Set sourceaddress   */
      BX_LOADERCMD_READWORD_A0D0(LoaderCode+LoaderLength,LoaderLength);                /* Read [A0] -> D0     */
      BX_LOADERCMD_LOADA1(LoaderCode+LoaderLength,BX_REG_COCO_DBI_RW,LoaderLength)     /* Set targetaddress   */
      BX_LOADERCMD_WRITEWORD_D0A1(LoaderCode+LoaderLength,LoaderLength);               /* Write D0 -> [A1]    */

      /* If reload needed:   Copy the image number (0,1,2 or 3) from Faust to COCO */    
      BX_LOADERCMD_LOADA0(LoaderCode+LoaderLength,BX_REG_SYS_RESETIMG_F,LoaderLength)  /* Set sourceaddress   */
      BX_LOADERCMD_READWORD_A0D0(LoaderCode+LoaderLength,LoaderLength);                /* Read [A0] -> D0     */
      BX_LOADERCMD_LOADA1(LoaderCode+LoaderLength,BX_REG_COCO_IMG_SELECT,LoaderLength) /* Set targetaddress   */
      BX_LOADERCMD_WRITEWORD_D0A1(LoaderCode+LoaderLength,LoaderLength);               /* Write D0 -> [A1]    */

      /* If reload needed: Initiate the reload by writing Coco register */
      BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_COCO_IMG_RELOAD,0x1,LoaderLength)

      /* If reload needed: Switch Mephisto LED off to indicate PU loader has stopped */
      BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_SYS_CTRL_REG,0x0,LoaderLength)

      /* If reload needed: Stop (PU loader is called again later, with bit 4 not set) */
      BX_LOADERCMD_END(LoaderCode+LoaderLength,0/* comment*/,LoaderLength) 

      /* No reload needed, if we get here */

      /* Check, wether an image reload has taken place */
      BX_LOADERCMD_READWORD(LoaderCode+LoaderLength,BX_REG_COCO_DBI_RW,LoaderLength)
      BX_LOADERCMD_COMPAREWORD(LoaderCode+LoaderLength,0x0010,0x0010,LoaderLength) /* Check bit 4 */

      /* Jump, if contents invalid (bit 4=0, i.e. no reload has taken place previously); continue if a reload has taken place */
      BX_LOADERCMD_BRANCHNOTEQUAL(LoaderCode+LoaderLength,BX_LOADER_IMAGE+LoaderLength+(4+12),LoaderLength)
      /* 4 bytes for BRANCHNOTEQUAL, 12 bytes for Coco/Faust copying */

      /* If reload had been done previously: Copy new PCI reset pattern from Coco to new Faust image. */
      BX_LOADERCMD_LOADA0(LoaderCode+LoaderLength,BX_REG_COCO_DBI_RW,LoaderLength)     /* Set sourceaddress   */
      BX_LOADERCMD_READWORD_A0D0(LoaderCode+LoaderLength,LoaderLength);                /* Read [A0] -> D0     */
      BX_LOADERCMD_LOADA1(LoaderCode+LoaderLength,BX_REG_SYS_RESETPATT_F,LoaderLength) /* Set targetaddress   */
      BX_LOADERCMD_WRITEWORD_D0A1(LoaderCode+LoaderLength,LoaderLength);               /* Write D0 -> [A1]    */

      /* We jump here, if no loading had been done previously */
    }

    /*************************************************************************
      Program relays (23er/30er only)
     *************************************************************************/

    /* We want to make sure, that the relays are in a valid position after PU loader has ran.
       This prevents (esp. 23ers: only PCI port !) from signaling an invalid relay status to the system,
       leading to a non booting system and thus a not accessible 23er.
       An initial BestXBoardProg() programs the (always correct shadow-) status of the relays to flash memory.
       We check for correct value and then make the PULoader use this value.
    */

    if (!BestXHasMephisto(handle))
    {
      /* Card has relays */

      /* Check, wether Flash already contains a valid relay value */
      BX_TRY(BestXDirectRegRead(handle,BX_RELAY_STATUS,sizeof(bx_int16),&val));
      if ((val&0xf0)==(0xf0& ~(val<<4))) /* Check complement (0xffff e.al. will surely fail here !) */
      {
        /* We should always go here.
           There seems to be a valid value in the flash already.
           We dare to set the relays from PULoader now....
           Rem: This has only an effect, if the relays have changed their position(s),
           e.g. during transport.
         */

        BX_LOADERCMD_LOADA0(LoaderCode+LoaderLength,BX_RELAY_STATUS,LoaderLength)      /* Set sourceaddress   */
        BX_LOADERCMD_READWORD_A0D0(LoaderCode+LoaderLength,LoaderLength);              /* Read [A0] -> D0     */
        BX_LOADERCMD_LOADA1(LoaderCode+LoaderLength,BX_REG_SYS_RELAY_REG,LoaderLength) /* Set targetaddress   */
        BX_LOADERCMD_WRITEWORD_D0A1(LoaderCode+LoaderLength,LoaderLength);             /* Write D0 -> [A1]    */
      }
      else
      {
        /* Flash not yet programmed for some reason.
           Should never happen (except during e.g. initial card production).
           You should execute a BestXBoardProg() once and then try again.
           We could return an error here (Rem: BestXBoardRead() returns an error in this case).
         */
      }
    }

    /*************************************************************************
      Stop the exerciser
     *************************************************************************/
    
     /* The run-bit is not resettet by PCIRST#, i.e. exerciser would start running 
       immediately again after switching to runmode.
     */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_EX_RUN_CTRL_REG,0x0,LoaderLength)
    
    /* Give HW some time to sync */
    BX_LOADERCMD_NOPWAIT(LoaderCode+LoaderLength,SYNCTIME,0 /* NOP's dummy */,LoaderLength)
    
    /* Force it. If synchronsiation has worked, forcing does no harm (SF) */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_EX_RUN_CTRL_FORCE_REG,0x0,LoaderLength)
    
    /* Init incrementor counter, indicating that PU loader has stopped the exerciser */
    DbgCntr++; 
    BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
    BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)
    
    /*************************************************************************
      Set the force retry bit
     *************************************************************************/

    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_FORCE_RETRY_BIT_REG,0x1,LoaderLength)
    
    /* Give HW some time to sync */
    BX_LOADERCMD_NOPWAIT(LoaderCode+LoaderLength,SYNCTIME,0 /* NOP's dummy */,LoaderLength)

    /* Force it. If synchronsiation has worked, forcing does no harm (SF) */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_FORCE_RETRY_SYNC_BIT_REG,0x0,LoaderLength)

    /* Init incrementor counter, indicating that PU loader has set the force retry bit */
    DbgCntr++; 
    BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
    BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)
    
    /*************************************************************************
      Switch exerciser into progmode (DBI clock) 
     *************************************************************************/

    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_EX_CLK_SWITCH_REG,0x0,LoaderLength)
    
    /* Give HW some time to sync */
    BX_LOADERCMD_NOPWAIT(LoaderCode+LoaderLength,SYNCTIME,0 /* NOP's dummy */,LoaderLength)

    /* Force it. If synchronsiation has worked, forcing does no harm (SF) */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_EX_CLK_SWITCH_STAT_REG,0x0,LoaderLength)
    
    /* Init incrementor counter, indicating that PU loader has switched into ProgMode */
    DbgCntr++; 
    BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
    BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)
    
    if (BestXHasMephisto(handle))
    {
      /*************************************************************************
        Init observer rules 
      *************************************************************************/

      /* Write the default values for obsrules SEM10 and SEM11 */
      /* SEM10: 12 bit HW register;
         A value of zero in HW (HW default:0xfff) means 0x1000
         A value of one would trigger the rule immediately */
      /* SEM11: 20 bit HW register;
         A value of zero in HW (HW default: undefined) means 0x100000
         A value of one would trigger the rule immediately */

      BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_SEM_10_PL_REG_M,0x0,LoaderLength)
      BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,BX_REG_SEM_11_PL_REG_M,0x0,LoaderLength)

      DbgCntr++; 
      BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
      BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)

      /*************************************************************************
        Init observer mask
      *************************************************************************/

      /* Caution: We have to write twice, because of a bug in mephisto */
      BX_TRY(BestXDirectRegRead(handle,BX_REG_PROT_ERR_MASK_M,sizeof(bx_int32),&mask));
      BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,BX_REG_PROT_ERR_MASK_M,mask,LoaderLength)
      BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,BX_REG_PROT_ERR_MASK_M,mask,LoaderLength)
      BX_TRY(BestXDirectRegRead(handle,BX_REG_PROT_ERR_MASK_M+4,sizeof(bx_int32),&mask));
      BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,BX_REG_PROT_ERR_MASK_M+4,mask,LoaderLength)
      BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,BX_REG_PROT_ERR_MASK_M+4,mask,LoaderLength)

      DbgCntr++; 
      BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
      BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)

      /*************************************************************************
        BX_REG_STAT_IAE_REG: Init interrupt status  (HW bugfix) 
      *************************************************************************/
      /* The shadowed interrupt status is only cleared by an explicit DBI write access.
        So after powerup this register is not initialized, i.e. contains
        random value (BX_STAT_INTA-D). PERR and SERR bits are also shadowed and
        not initialized by HW.
      */

      /* Init/reset to zero bits 4-9 */
      BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_STAT_IAE_REG,0x3f0,LoaderLength)

      DbgCntr++; 
      BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
      BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)

    } /* HasMephisto */
    else
    {
      /*************************************************************************
        Init observer mask
      *************************************************************************/

      BX_TRY(BestXDirectRegRead(handle,BX_REG_PROT_ERR_MASK_F,sizeof(bx_int32),&mask));
      mask &= ~(1UL<<10); /* Disable/mask rule TARGET_6 (buggy in Faust HW) */
      BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,BX_REG_PROT_ERR_MASK_F,mask,LoaderLength)

      BX_TRY(BestXDirectRegRead(handle,BX_REG_PROT_ERR_MASK_F+4,sizeof(bx_int32),&mask));
      BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,BX_REG_PROT_ERR_MASK_F+4,mask,LoaderLength)
    }
    
      /*************************************************************************
        BX_REG_MSM_STAT_MON: Init Abort status  (HW bugfix) 
      *************************************************************************/
      /* The Abort status is only cleared by an explicit DBI write access.
        After powerup this register is not initialized, i.e. contains
        random value. This bug is still present in E2930A and E2930B (first cut).
      */

      /* Init/reset to zero */
      BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_MSM_STAT_MON,0x0,LoaderLength)

      DbgCntr++; 
      BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
      BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)


    /*************************************************************************
      Init config space
     *************************************************************************/

    /* Get LoaderCode for config space and decoders */
    BX_TRY(BestXTConfigPUCodeGet(handle,LoaderCode,&LoaderLength));
    DbgCntr++;
    BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
    BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)

    /* Set first DW of extended Faust config space memory to zero to indicate the absence of 
       any extended Capability list items */
    if (BestXHasFaust(handle)) /* strongly needed for Faust-ASIC only */
    {
      /* Set address-offset into (PCI) config space */
      BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,BX_REG_CNF_MEM_ADDR_F,0x100,LoaderLength)
      /* Set data */
      BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,BX_REG_CNF_MEM_DATA_F,0x0,LoaderLength)
    }

    /*************************************************************************
      Init decoders
     *************************************************************************/
      
    BX_TRY(BestXTDecoderProg(handle,LoaderCode,&LoaderLength));

    DbgCntr++;
    BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
    BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)
 
    /*************************************************************************
      Init exerciser memories
     *************************************************************************/

    /* Caution: The autoincrement feature does not work as long as RST# is asserted,
                so we can program one line only. */


    /*************************************************************************
      Completer Target
     *************************************************************************/
                
    /* Completer Target behavior */
    /* MIN-MAX col */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_TBEHAV_MIN_REG,0x0,LoaderLength)
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_TBEHAV_MAX_REG,BX_CTBEH_MEMWIDTH-1,LoaderLength)

    /* Starting row */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_TBEHAV_CTR,0x0,LoaderLength)
    
    /* Data */
    /* Mem 0: REPEAT forever */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_TBEHAV_DATA_REG,0x0,LoaderLength)

    /* Mem1: LATENCY=3, INITIAL=0 (i.e. accept) */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_TBEHAV_DATA_REG,3|0<<5,LoaderLength)
    
    /* Mem2: SUBSEQPHASE#0=0 */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_TBEHAV_DATA_REG,0x0,LoaderLength)
    
    /* Mem3: SUBSEQPHASE#1=0, SUBSEQ=0, SPLITENABLE=0  (i.e. accept) */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_TBEHAV_DATA_REG,0|0<<4|0<<6,LoaderLength)
    
    /* Mem4: SPLITLATENCY=3, DECSPEED=1 (i.e. B), ACK64=1*/
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_TBEHAV_DATA_REG,3|1<<4|1<<6,LoaderLength)
    

    /* Split condition properties */
    /* Mask LO */
    BX_TRY(BestXCTSplitCondGet(handle,0,BX_CTSPLIT_ADDRMASK_LO,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRMASK_LO_REG+0*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,1,BX_CTSPLIT_ADDRMASK_LO,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRMASK_LO_REG+1*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,2,BX_CTSPLIT_ADDRMASK_LO,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRMASK_LO_REG+2*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,3,BX_CTSPLIT_ADDRMASK_LO,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRMASK_LO_REG+3*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)

    /* Mask HI */
    BX_TRY(BestXCTSplitCondGet(handle,0,BX_CTSPLIT_ADDRMASK_HI,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRMASK_HI_REG+0*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,1,BX_CTSPLIT_ADDRMASK_HI,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRMASK_HI_REG+1*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,2,BX_CTSPLIT_ADDRMASK_HI,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRMASK_HI_REG+2*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,3,BX_CTSPLIT_ADDRMASK_HI,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRMASK_HI_REG+3*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)

    /* Value LO */
    BX_TRY(BestXCTSplitCondGet(handle,0,BX_CTSPLIT_ADDRVAL_LO,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRVAL_LO_REG+0*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,1,BX_CTSPLIT_ADDRVAL_LO,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRVAL_LO_REG+1*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,2,BX_CTSPLIT_ADDRVAL_LO,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRVAL_LO_REG+2*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,3,BX_CTSPLIT_ADDRVAL_LO,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRVAL_LO_REG+3*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)

    /* Value HI */
    BX_TRY(BestXCTSplitCondGet(handle,0,BX_CTSPLIT_ADDRVAL_HI,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRVAL_HI_REG+0*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,1,BX_CTSPLIT_ADDRVAL_HI,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRVAL_HI_REG+1*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,2,BX_CTSPLIT_ADDRVAL_HI,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRVAL_HI_REG+2*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,3,BX_CTSPLIT_ADDRVAL_HI,&val)); 
    BX_LOADERCMD_WRITELONG(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_ADDRVAL_HI_REG+3*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    
    /* Commands */
    BX_TRY(BestXCTSplitCondGet(handle,0,BX_CTSPLIT_CMDS,&val)); 
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_CMD_REG+0*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,1,BX_CTSPLIT_CMDS,&val)); 
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_CMD_REG+1*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,2,BX_CTSPLIT_CMDS,&val)); 
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_CMD_REG+2*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,3,BX_CTSPLIT_CMDS,&val)); 
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_CMD_REG+3*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)

    /* Decoders */
    BX_TRY(BestXCTSplitCondGet(handle,0,BX_CTSPLIT_DEC,&val)); 
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_DEC_REG+0*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,1,BX_CTSPLIT_DEC,&val)); 
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_DEC_REG+1*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,2,BX_CTSPLIT_DEC,&val)); 
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_DEC_REG+2*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,3,BX_CTSPLIT_DEC,&val)); 
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_DEC_REG+3*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
      
    /* Queues */
    BX_TRY(BestXCTSplitCondGet(handle,0,BX_CTSPLIT_QUEUE,&val)); 
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_QUEUE_REG+0*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,1,BX_CTSPLIT_QUEUE,&val)); 
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_QUEUE_REG+1*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,2,BX_CTSPLIT_QUEUE,&val)); 
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_QUEUE_REG+2*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
    BX_TRY(BestXCTSplitCondGet(handle,3,BX_CTSPLIT_QUEUE,&val)); 
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,(BX_REG_TSPLIT0_QUEUE_REG+3*BX_SPLITCOND_ADDR_OFFS),val,LoaderLength)
 
    DbgCntr++;
    BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
    BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)

    /*************************************************************************
      STOP in case of 29er/30er
     *************************************************************************/
    
    /* Write 'PRdy' to display */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_DISP_DIGIT3,0x50,LoaderLength)
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_DISP_DIGIT2,0x52,LoaderLength)
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_DISP_DIGIT1,0x64,LoaderLength)
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_DISP_DIGIT0,0x79,LoaderLength)

    /* Check HW type ("E2929A" resp. "E2930A" or "E2922A" resp. E2923A") 
      A 29er/30er will stop here whereas a 22er/23er waits for Mephisto's delayed PCIRST# signal
      to be deasserted (maybe forever!) and will then switch into RunMode 
    */

    BX_LOADERCMD_READWORD(LoaderCode+LoaderLength,BX_OVERLOADED_PRODUCT_STRING+4,LoaderLength)
    if (BestXHasMephisto(handle))
    {
      /* Check for "E2929A" */
      BX_LOADERCMD_COMPAREWORD(LoaderCode+LoaderLength,0x3900,0xff00,LoaderLength) /* 0x39='9' */
    }
    else
    {
      /* Check for "E2930A" */
      BX_LOADERCMD_COMPAREWORD(LoaderCode+LoaderLength,0x3000,0xff00,LoaderLength) /* 0x30='0' */
    }
      
    /* Jump in case of 22er/23er; continue in case of 29er/30er */
    BX_LOADERCMD_BRANCHNOTEQUAL(LoaderCode+LoaderLength,BX_LOADER_IMAGE+LoaderLength+(4+6+2),LoaderLength)
    /* 4 bytes for BRANCHNOTEQUAL, 6 bytes for WRITEWORD, 2 bytes for END */

    /* 29er/30er only: Switch Mephisto LED off to indicate PU loader has stopped */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_SYS_CTRL_REG,0x0,LoaderLength)

    /* End for 29er/30er here; to be continued by Firmcore... */
    BX_LOADERCMD_END(LoaderCode+LoaderLength,29/* comment*/,LoaderLength) 
    
    /*************************************************************************
      Only 22ers will ever get here (or if card's flash is not progammed correctly) 
     *************************************************************************/

    DbgCntr+=10; /* Increment, indicating that 29/30er has finished its code */
    BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
    BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)

    /*************************************************************************
      Wait until the internally delayed  PCIRST# has been deasserted (22/23er only)
     *************************************************************************/
    
    /* This is bit 7 of BX_REG_SYS_STAT_MONITOR.
       Remark: If you switch into Runmode during the 100ms after PCIRST# 
       has been deasserted, you will potentially loose Block and Behavior 
       memory contents. For this, we wait for the internally delayed PCIRST#
       signal here. 
       Caution: If we are not plugged into a slot, the 
       internally delayed PCIRST# signal remains asserted forever !!
       
       (PCI power not good) OR (PCIRST# asserted) => internally delayed PCIRST# signal is asserted
        
       Except the 100ms after the deassertion of PCIRST#, '<=' is also true.
    */

    /* Write 'RST#' to display to indicate that we are waiting for iRST# */
    /* Remark: 22/23er has no display and 29/30er will never get here.
       So when you see this on the display surely the FLASH memory of a 29/30er board
       is not programmed. 
    */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_DISP_DIGIT3,0x52,LoaderLength)
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_DISP_DIGIT2,0x53,LoaderLength)
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_DISP_DIGIT1,0x54,LoaderLength)
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_DISP_DIGIT0,0x23,LoaderLength)

    BX_LOADERCMD_READLONG(LoaderCode+LoaderLength,BX_REG_SYS_STAT_MONITOR,LoaderLength)
    BX_LOADERCMD_COMPARELONG(LoaderCode+LoaderLength,0,1<<7,LoaderLength)
    /* Jump back to BX_LOADERCMD_READLONG, if bit 7 equals 0 (i.e. RST# is still asserted) */
    BX_LOADERCMD_BRANCHEQUAL(LoaderCode+LoaderLength,BX_LOADER_IMAGE+LoaderLength-(4+10),LoaderLength)
    /* Remark: 4 is #bytes for BX_LOADERCMD_READLONG; 10 is #bytes for BX_LOADERCMD_COMPARELONG */    

    DbgCntr++; /* Increment by one, indicating that PCIRST# has been deasserted */
    BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
    BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)
        
    /*************************************************************************
      Switch exerciser into runmode (PCI clock) (22/23er only)
     *************************************************************************/

    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_EX_CLK_SWITCH_REG,0x1,LoaderLength)
    
    /* Give HW some time to sync */
    BX_LOADERCMD_NOPWAIT(LoaderCode+LoaderLength,SYNCTIME,0 /* NOP's dummy */,LoaderLength)

    /* Force it. If synchronsiation has worked, forcing does no harm (SF) */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_EX_CLK_SWITCH_STAT_REG,0x0,LoaderLength)
    
    /* Init incrementor counter, indicating that PU loader has switched into RunMode */
    DbgCntr++; 
    BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
    BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)
    
    /*************************************************************************
      Switch analyzer into runmode (PCI clock) (22/23er only)
     *************************************************************************/

    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_AN_CLK_SWITCH_REG,0x1,LoaderLength)
    
    /* Give HW some time to sync */
    BX_LOADERCMD_NOPWAIT(LoaderCode+LoaderLength,SYNCTIME,0 /* NOP's dummy */,LoaderLength)

    /* Force it. If synchronsiation has worked, forcing does no harm (SF) */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_AN_CLK_SWITCH_STAT_REG,0x0,LoaderLength)
    
    /* Init incrementor counter, indicating that PU loader has switched into RunMode */
    DbgCntr++; 
    BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
    BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)
    
    /*************************************************************************
      Switch trace into runmode (PCI clock) (22/23er only)
     *************************************************************************/

    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_TR_CLK_SWITCH_REG,0x1,LoaderLength)
    
    /* Give HW some time to sync */
    BX_LOADERCMD_NOPWAIT(LoaderCode+LoaderLength,SYNCTIME,0 /* NOP's dummy */,LoaderLength)

    /* Force it. If synchronsiation has worked, forcing does no harm (SF) */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_TR_CLK_SWITCH_STAT_REG,0x0,LoaderLength)
    
    /* Init incrementor counter, indicating that PU loader has switched into RunMode */
    DbgCntr++; 
    BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
    BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)
    
    /*************************************************************************
      Clear the force retry bit (22/23er only)
     *************************************************************************/

    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_FORCE_RETRY_BIT_REG,0x0,LoaderLength)
    
    /* Give HW some time to sync */
    BX_LOADERCMD_NOPWAIT(LoaderCode+LoaderLength,SYNCTIME,0 /* NOP's dummy */,LoaderLength)

    /* Force it. If synchronsiation has worked, forcing does no harm (SF) */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_FORCE_RETRY_SYNC_BIT_REG,0x0,LoaderLength)

    /* Init incrementor counter, indicating that PU loader has cleared the force retry bit */
    DbgCntr++; 
    BX_LOADERCMD_MASKWRITELONG(LoaderCode+LoaderLength,BX_REG_RES_LOCK_REG,DbgCntr,BX_PU_COUNTERMASK,LoaderLength)
    BX_LOADERCMD_COMMENT(LoaderCode+LoaderLength,DbgCntr,LoaderLength)

    /*************************************************************************
      STOP (22/23er only)
     *************************************************************************/

    /* Write 'PRdy' to display (usually never visible) */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_DISP_DIGIT3,0x50,LoaderLength)
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_DISP_DIGIT2,0x52,LoaderLength)
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_DISP_DIGIT1,0x64,LoaderLength)
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_DISP_DIGIT0,0x79,LoaderLength)

    /* 22/23er only: Switch Mephisto LED off to indicate PU loader has stopped */
    BX_LOADERCMD_WRITEWORD(LoaderCode+LoaderLength,BX_REG_SYS_CTRL_REG,0x0,LoaderLength)


    /* Stop the stupid loader from running into LaLa land.
       This must be the final loader command !!! */
    BX_LOADERCMD_END(LoaderCode+LoaderLength,22,LoaderLength) /* 22/23er only */

    /*************************************************************************
     *************************************************************************
     ************************* END OF PU-CODE ********************************
     *************************************************************************
     *************************************************************************/

    if (LoaderLength>BX_LOADER_CODELENGTH)
    {
      /* Should never happen, because the number of bytes written into
         array LoaderCode[] is constant.
      */
      BX_TRY_FAIL(BX_E_INVALID_CASE);
    }
    *LoaderLengthPtr=LoaderLength;
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXPURead(
 *
 * Purpose: Reads current power-up loadre coder from card and converts
 *          it into the current settings.
 * Status:  Not yet implemented.
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXPURead(
  bx_handletype handle
)
{
  BX_ERRETURN(BX_E_TBD);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXPURestore
 *
 * Purpose: Restores the power up settings by executing the power-up
 *          loader code from flash memory. Mainly for test purposes.
 * CAUTION: This call may lead to inconsistencies between the current
 *          card's settings and the contents of the host database !
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXPURestore(
  bx_handletype handle
)
{
  BX_DECLARE_FUNCNAME("BestXPURestore");
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* Restart PU loader */
    BX_TRY(BestXDirectRegWrite( handle,
                                BX_REG_DEFLOAD_RESTART_STRB,
                                sizeof(bx_int16),   /*regsize*/
                                0x0                 /*regvalue*/
                              ));
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXLoaderDecode()
 *
 * Purpose: Writes the current power-up commands into a file        
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXLoaderDecode(
  bx_handletype handle,
  char* filename,           /* Report filename              */
  bx_int32 flashaddr        /* Start of Loadercode in Flash */
)
{
  BX_DECLARE_FUNCNAME("BestXLoaderDecode");
  BX_TRY_VARS_NO_PROG;

  FILE *fp;                      /* Output filename pointer                */
  bx_int16 *CurAddr;             /* current address of loadercode in flash */
  bx_int32 Opcode;               /* Current Command                        */
  bx_int32 Param[6];             /* Optional command parameters            */

  /* These are extracted from Param[] */
  bx_int32 Address;
  bx_int32 Data;
  bx_int32 Mask;
  bx_int32 Repeat;
  bx_int32 ContinueDecoding=2;   /* 2 consequtive LOADER_END commands will stop the decoding */
  
  const bx_int32 MAX_OUT_LEN=20; /* field width for BESTX_FPRINTF()              */

  /* Try to write to a file */
  if (filename==NULL || ((fp=fopen(filename,"w"))==NULL))
  {
    fp = stdout;
  }
    
  BESTX_FPRINTF(fp,"/******************************************************************\n");
  BESTX_FPRINTF(fp," *                       Powerup loader code                      *\n");
  BESTX_FPRINTF(fp," ******************************************************************/\n\n");

  BX_TRY_BEGIN
  {
    if (BestXIsOffline(handle))
    {
      BESTX_FPRINTF(fp," ******************************************************************/\n");
      BESTX_FPRINTF(fp," ************************  OFFLINE  *******************************/\n");
      BESTX_FPRINTF(fp," ******************************************************************/\n");
      if (fp)
      {
        fclose(fp);
      }
      BX_ERRETURN(BX_E_OK);
    }

    CurAddr = (bx_int16ptr)((bx_intptr) flashaddr); /* Here starts the loader code in flash */
    do
    {
      BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+0));
      Opcode   = (Param[0]&0xff00)>>8;  /* Command         */
      Param[0] =  Param[0]&0xff;        /* First parameter */

      switch (Opcode)
      {
        case BX_LOADER_WRITELONG:
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+2));
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+3));
          Address=(Param[0]<<16) | Param[1];
          Data=(Param[2]<<16) | Param[3];
          BESTX_FPRINTF(fp,"0x%06lx %-*s addr:0x%06lx data:0x%08lx\n",
                  CurAddr-4,MAX_OUT_LEN,"WRITELONG",Address,Data);
          break;

        case BX_LOADER_WRITEWORD: 
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+2));
          Address=(Param[0]<<16) | Param[1];
          Data=Param[2];
          BESTX_FPRINTF(fp,"0x%06lx %-*s addr:0x%06lx data:0x%08lx\n",
                  CurAddr-3,MAX_OUT_LEN,"WRITEWORD",Address,Data);
          break;

        case BX_LOADER_WRITEBYTE: 
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+2));
          Address=(Param[0]<<16) | Param[1];
          Data=Param[2]&0xff;
          BESTX_FPRINTF(fp,"0x%06lx %-*s addr:0x%06lx data:0x%08lx\n",
                  CurAddr-3,MAX_OUT_LEN,"WRITEBYTE",Address,Data);
          break;

        case BX_LOADER_READLONG:
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          Address=(Param[0]<<16) | Param[1];
          BESTX_FPRINTF(fp,"0x%06lx %-*s addr:0x%06lx\n",
                  CurAddr-2,MAX_OUT_LEN,"READLONG",Address);
          break;

        case BX_LOADER_READWORD:
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          Address=(Param[0]<<16) | Param[1];
          BESTX_FPRINTF(fp,"0x%06lx %-*s addr:0x%06lx\n",
                  CurAddr-2,MAX_OUT_LEN,"READWORD",Address);
          break;

        case BX_LOADER_READBYTE:
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          Address=(Param[0]<<16) | Param[1];
          BESTX_FPRINTF(fp,"0x%06lx %-*s addr:0x%06lx\n",
                  CurAddr-2,MAX_OUT_LEN,"READBYTE",Address);
          break;

        case BX_LOADER_MASKWRITELONG: 
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+2));
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+3));
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+4));
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+5));
          Address=(Param[0]<<16) | Param[1];
          Data=(Param[2]<<16) | Param[3];
          Mask=(Param[4]<<16) | Param[5];
          BESTX_FPRINTF(fp,"0x%06lx %-*s addr:0x%06lx data:0x%08lx mask:0x%08lx\n",
                  CurAddr-6,MAX_OUT_LEN,"MASKWRITELONG",Address,Data,Mask);
          break;

        case BX_LOADER_MASKWRITEWORD: 
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+2));
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+3));
          Address=(Param[0]<<16) | Param[1];
          Data=Param[2];
          Mask=Param[3];
          BESTX_FPRINTF(fp,"0x%06lx %-*s addr:0x%06lx data:0x%08lx mask:0x%08lx\n",
                  CurAddr-4,MAX_OUT_LEN,"MASKWRITEWORD",Address,Data,Mask);
          break;

        case BX_LOADER_MASKWRITEBYTE:
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+2));
          Address=(Param[0]<<16) | Param[1];
          Data=(Param[2]&0xff00)>>16;
          Mask=Param[2] & 0xff;
          BESTX_FPRINTF(fp,"0x%06lx %-*s addr:0x%06lx data:0x%08lx mask:0x%08lx\n",
                  CurAddr-3,MAX_OUT_LEN,"MASKWRITEBYTE",Address,Data,Mask);
          break;

        case BX_LOADER_COMPARELONG:
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+2));
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+3));
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+4));
          Data=(Param[1]<<16) | Param[2];
          Mask=(Param[3]<<16) | Param[4];
          BESTX_FPRINTF(fp,"0x%06lx %-*s data:0x%08lx mask:0x%08lx\n",
                  CurAddr-5,MAX_OUT_LEN,"COMPARELONG",Data,Mask);
          break;

        case BX_LOADER_COMPAREWORD:
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+2));
          Data=Param[1];
          Mask=Param[2];
          BESTX_FPRINTF(fp,"0x%06lx %-*s data:0x%08lx mask:0x%08lx\n",
                  CurAddr-3,MAX_OUT_LEN,"COMPAREWORD",Data,Mask);
          break;

        case BX_LOADER_COMPAREBYTE:
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          Data=(Param[1]&0xff00)>>16;
          Mask=Param[1] & 0xff;
          BESTX_FPRINTF(fp,"0x%06lx %-*s data:0x%08lx mask:0x%08lx\n",
                  CurAddr-2,MAX_OUT_LEN,"COMPAREBYTE",Data,Mask);
          break;

        case BX_LOADER_BRANCHEQUAL:
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          Address=(Param[0]<<16) | Param[1];
          BESTX_FPRINTF(fp,"0x%06lx %-*s addr:0x%06lx\n",
                  CurAddr-2,MAX_OUT_LEN,"BRANCHEQUAL",Address);
          break;

        case BX_LOADER_BRANCHNOTEQUAL:    
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          Address=(Param[0]<<16) | Param[1];
          BESTX_FPRINTF(fp,"0x%06lx %-*s addr:0x%06lx\n",
                  CurAddr-2,MAX_OUT_LEN,"BRANCHNOTEQUAL",Address);
          break;

        case BX_LOADER_DONTIMESBYTE:
          Repeat=Param[0];
          BESTX_FPRINTF(fp,"0x%06lx %-*s N:%lu\n",
                  CurAddr-1,MAX_OUT_LEN,"DONTIMESBYTE",Repeat);
          break;

        case BX_LOADER_DONTIMESWORD:
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          Repeat=Param[1];
          BESTX_FPRINTF(fp,"0x%06lx %-*s N:%lu\n",
                  CurAddr-2,MAX_OUT_LEN,"DONTIMESWORD",Repeat);
          break;

        case BX_LOADER_ENDDONTIMES:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"ENDDONTIMES");
          break;

        case BX_LOADER_NOP:
          BESTX_FPRINTF(fp,"0x%06lx %-*s Comment:%u\n",
                  CurAddr-1,MAX_OUT_LEN,"NOP",Param[0]);
          break;

        case BX_LOADER_END:
          BESTX_FPRINTF(fp,"0x%06lx %-*s Comment:%u\n",
            CurAddr-1,MAX_OUT_LEN,"PROGRAM END",Param[0]);
          break;

        case BX_LOADER_READLONG_A0D0:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"READLONG: [A0]->D0");
          break;

        case BX_LOADER_READWORD_A0D0:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"READWORD: [A0]->D0");
          break;

        case BX_LOADER_READBYTE_A0D0:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"READBYTE: [A0]->D0");
          break;

        case BX_LOADER_READLONG_A0D0_INCR:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"READLONG: [A0++]->D0");
          break;

        case BX_LOADER_READWORD_A0D0_INCR:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"READWORD: [A0++]->D0");
          break;

        case BX_LOADER_READBYTE_A0D0_INCR:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"READBYTE: [A0++]->D0");
          break;

        case BX_LOADER_READLONG_A0D1:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"READLONG: [A0]->D1");
          break;

        case BX_LOADER_READWORD_A0D1:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"READWORD: [A0]->D1");
          break;

        case BX_LOADER_READBYTE_A0D1:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"READBYTE: [A0]->D1");
          break;

        case BX_LOADER_READLONG_A0D1_INCR:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"READLONG: [A0++]->D1");
          break;

        case BX_LOADER_READWORD_A0D1_INCR:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"READWORD: [A0++]->D1");
          break;

        case BX_LOADER_READBYTE_A0D1_INCR:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"READBYTE: [A0++]->D1");
          break;

        case BX_LOADER_WRITELONG_D0A1:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"WRITELONG: D0->[A1]");
          break;

        case BX_LOADER_WRITEWORD_D0A1:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"WRITEWORD: D0->[A1]");
          break;

        case BX_LOADER_WRITEBYTE_D0A1:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"WRITEBYTE: D0->[A1]");
          break;

        case BX_LOADER_WRITELONG_D0A1_INCR:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"WRITELONG: D0->[A1++]");
          break;

        case BX_LOADER_WRITEWORD_D0A1_INCR:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"WRITEWORD: D0->[A1++]");
          break;

        case BX_LOADER_WRITEBYTE_D0A1_INCR:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"WRITEBYTE: D0->[A1++]");
          break;

        case BX_LOADER_WRITELONG_D1A1:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"WRITELONG: D1->[A1]");
          break;

        case BX_LOADER_WRITEWORD_D1A1:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"WRITEWORD: D1->[A1]");
          break;

        case BX_LOADER_WRITEBYTE_D1A1:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"WRITEBYTE: D1->[A1]");
          break;

        case BX_LOADER_WRITELONG_D1A1_INCR:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"WRITELONG: D1->[A1++]");
          break;

        case BX_LOADER_WRITEWORD_D1A1_INCR:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"WRITEWORD: D1->[A1++]");
          break;

        case BX_LOADER_WRITEBYTE_D1A1_INCR:
          BESTX_FPRINTF(fp,"0x%06lx %-*s\n",
                  CurAddr-1,MAX_OUT_LEN,"WRITEBYTE: D1->[A1++]");
          break;

        case BX_LOADER_LOADA0:
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          Address=(Param[0]<<16) | Param[1];
          BESTX_FPRINTF(fp,"0x%06lx %-*s addr:0x%06lx\n",
                  CurAddr-2,MAX_OUT_LEN,"LOAD A0",Address);
          break;

        case BX_LOADER_LOADA1:
          BX_TRY(BestXDirectRegRead(handle,(bx_int32)CurAddr++,sizeof(bx_int16),Param+1));
          Address=(Param[0]<<16) | Param[1];
          BESTX_FPRINTF(fp,"0x%06lx %-*s addr:0x%06lx\n",
                  CurAddr-2,MAX_OUT_LEN,"LOAD A1",Address);
          break;

        default:
          BESTX_FPRINTF(fp,
                  "ERROR: Unknown opcode (0x%02lx) at address 0x%06lx (aborted)\n",
                  Opcode,CurAddr-1);
          BX_E_ERROR_MSG_SET("Unknown opcode");
          BX_TRY_FAIL(BX_E_ERROR);
          break;
      }  /* switch */

      if (Opcode==BX_LOADER_END)
      {
        /* Last command was a LOADER_END command */
        ContinueDecoding--;
      }
      else
      {
        /* 2 consequtive LOADER_END commands will stop the decoding */
        ContinueDecoding=2;
      }

    } while(ContinueDecoding);
  } /* TRY */

  if (fp)
  {
    fclose(fp);
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

#if 0 
/* Here are the Testfunktion for the Powerup loeader please don't remove if 0!!
*/
/******************************************************************************
*                                                                             *
*                Testfunktion for Powerup loader code                         *
*                                                                             *
* If you use this Testfunktion, please select the funktion                    *
* BestXDirectRegRead and replace that funktion with this funktion             *
*                 Here are only static values
*                                                                             *
*******************************************************************************/
/* values :

  first A -> Address line             second A -> high byte
  first D -> Data line                second B -> low Byte
  first C -> Compare Data             
  first E -> Mask pattern             unused -> FF
*/
bx_errtype EXPORT BestXDirectRegRead(
  bx_handletype handle,
  bx_int32 dir_addr,
  bx_int32 regsize,
  bx_int32 * reg_value
)
{
  bx_int16 fld [76] = 
  {
    0x82AB,0xAAAB,0xDADB,                       /*0x000a0000 WRITEWORD addr:0x00abaaab data:0x0000dadb*/
    0x80AB,0xAAAB,0xDADB,0xDADB,                /*0x000a0006 WRITELONG addr:0x00abaaab data:0xdadbdadb*/
    0x81AB,0xAAAB,0xFFDB,                       /*0x000a000e WRITEBYTE addr:0x00abaaab data:0x000000db*/
    0x02AB,0xAAAB,                              /*0x000a0014 READWORD  addr:0x00abaaab*/
    0x00AB,0xAAAB,                              /*0x000a0018 READLONG  addr:0x00abaaab*/
    0x01AB,0xAAAB,                              /*0x000a001c READBYTE  addr:0x00abaaab*/
    0x42AB,0xAAAB,0xDADB,0xEAEB,                /*0x000a0020 MASKWRITEWORD addr:0x00abaaab data:0x0000dadb Mask:0x0000eaeb*/
    0x40AB,0xAAAB,0xDADB,0xDADB,0xEAEB,0xEAEB,  /*0x000a0028 MASKWRITELONG addr:0x00abaaab data:0xdadbdadb Mask:0xeaebeaeb*/
    0x41AB,0xAAAB,0xDAEB,                       /*0x000a0034 MASKWRITEBYTE addr:0x00abaaab data:0x00000000 Mask:0x000000eb*/
    0xD2FF,0xCACB,0xEAEB,                       /*0x000a003a COMPAREWORD  data:0x0000cacb Mask:0x0000eaeb*/
    0xD0FF,0xCACB,0xCACB,0xEAEB,0xEAEB,         /*0x000a0040 COMPARELONG data:0xcacbcacb Mask:0xeaebeaeb*/
    0xD1FF,0xCAEB,                              /*0x000a004a COMPAREBYTE data:0x00000000 Mask:0x000000eb*/
    0xE3AB,0xAAAB,                              /*0x000a004e BRANCHEQUAL addr:0x00abaaab*/
    0xE7AB,0xAAAB,                              /*0x000a0052 BRANCHNOTEQUAL addr:0x00abaaab*/
    0xC3FB,                                     /*0x000a0056 DONTIMES   N:251*/
    0xC7FF,                                     /*0x000a0058 ENDDONTIMES */
    0xEBFF,                                     /*0x000a005a NOP*/    
    0x10FF,                                     /*READLONG_A0D0*/
    0x12FF,                                     /*READWORD_A0D0*/
    0x11FF,                                     /*READBYTE_A0D0*/
    0x20FF,                                     /*READLONG_A0D0_INCR*/
    0x22FF,                                     /*READWORD_A0D0_INCR*/
    0x21FF,                                     /*READBYTE_A0D0_INCR*/
    0x18FF,                                     /*READLONG_A0D1*/
    0x1AFF,                                     /*READWORD_A0D1*/
    0x19FF,                                     /*READBYTE_A0D1*/
    0x28ff,                                     /*READLONG_A0D1_INCR*/
    0x2AFF,                                     /*READWORD_A0D1_INCR*/
    0x29FF,                                     /*READBYTE_A0D1_INCR*/
    0x94FF,                                     /*WRITELONG_D0A1*/
    0x96FF,                                     /*WRITEWORD_D0A1*/
    0x95FF,                                     /*WRITEBYTE_D0A1*/
    0xA4FF,                                     /*WRITELONG_D0A1_INCR*/
    0xA6FF,                                     /*WRITEWORD_D0A1_INCR*/
    0xA5FF,                                     /*WRITEBYTE_D0A1_INCR*/
    0x9CFF,                                     /*WRITELONG_D1A1*/
    0x9EFF,                                     /*WRITEWORD_D1A1*/
    0x9DFF,                                     /*WRITEBYTE_D1A1*/
    0xACFF,                                     /*WRITELONG_D1A1_INCR*/
    0xAEFF,                                     /*WRITEWORD_D1A1_INCR*/
    0xADFF,                                     /*WRITEBYTE_D1A1_INCR*/
    0x30AB, 0xAAAB,                             /*LOADA0*/
    0x34AB, 0xAAAB,                             /*LOADA1*/
    0xFFFF                                      /*0x000a005c PROGRAM END*/
  };

  dir_addr = ((dir_addr - 0x0a0000)/2);
  *reg_value = fld[dir_addr];

  BX_ERRETURN(BX_E_OK);
}
#endif

/* -----------------------------------------------------------------
 * PME Bit access Functions
 * ----------------------------------------------------------------- */

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXPMEWrite(
 *
 * Purpose      : Sets or resets the Power Management enable bit
 * Inputs       : Value of PME-Bit
 * Remarks      : PMEWrite works only with pull-up resistor on back-plane 
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXPMEWrite(
    bx_handletype handle,
    bx_int32 value
)
{
  BX_DECLARE_FUNCNAME("BestXPMEWrite [pmewrite]");
  BX_TRY_VARS_NO_PROG;
  bx_int32 oldvalue=0;
  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_CAPI);
    BX_E_ERROR_MSG_SET("This command is not supported by the E2940A CompactPCI card");
    BX_TRY_FAIL(BestXIsCompact(handle) ? BX_E_ERROR : BX_E_OK);
    BX_TRY_FCT_PARAM(1, value > 1UL);
    BX_TRY(BestXDirectRegRead(handle,BX_PCI_REG,sizeof(bx_int16),&oldvalue));
    if (value)
    {
      value=oldvalue | 0x1; /* Set bit 0 */
    }
    else
    {
      value=oldvalue & 0xfffffffe; /* Clear bit 0 */
    }
    
    BX_TRY(BestXDirectRegWrite(handle,BX_PCI_REG,sizeof(bx_int16),value));
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXPMERead(
 *
 * Purpose      : Reads the value of theb Power Management Enable bit
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXPMERead(
    bx_handletype handle,
    bx_int32 * val
)
{
  BX_DECLARE_FUNCNAME("BestXPMERead [pmeread]");

  BX_TRY_VARS_NO_PROG;
  
  BX_FCT_PARAM_NULL_POINTER_CHK(val);
  
  BX_TRY_BEGIN
  {
    BX_TRY_LICENSE(BX_CAPABILITY_CAPI);
    
    BX_TRY(BestXDirectRegRead(  handle,
                                BX_PCI_REG,
                                sizeof(bx_int16),   /*regsize*/
                                val                 /*regvalue*/
                              ));
    
    *val&=0x1;
  }

  BX_ERRETURN(BX_TRY_RET);
}



/*---------------------------------------------------------------------------*
 * BestXInterruptGenerate()
 *
 * Purpose:  Generates PCI interrupts A,B,C and/or D   
 *           (Compare xtypedef.h:
 *            BX_INTA 0x1
 *            BX_INTB 0x2
 *            BX_INTC 0x4
 *            BX_INTD 0x8
 *                        )
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXInterruptGenerate( 
  bx_handletype handle,
  bx_int32  pci_int
)
{
  BX_DECLARE_FUNCNAME("BestXInterruptGenerate [interruptgenerate]");
  
  BX_TRY_VARS_NO_PROG;
 
  BX_TRY_BEGIN {
    bx_int32 intNum = 0;        /* 0==inta .. 3==intd */
    bx_int32 intMask = pci_int;
    
    /* It is allowed to generate multiple interrupts simultaneously */
    BX_FCT_PARAM_CHK(2,pci_int>0xf);
  
    /* several tasks have to be performed here. */
    /* 1. for all interrupts that are to be asserted, set HW counter */
    /*    to one (assert after one clock) */
    /* 2. enable the interrupts in the mask register */
    /* 3. set assert reg to 0: interrupt assertion is edge-sensitive */
    /* 4. assert the desired interrupts */

    /* 1. */
    while (intMask)
    {
      if (intMask & 0x1UL)
      {
        BX_TRY(BestXDirectRegWrite(handle,
                                   BX_REG_INTA_CLK_CNT_REG + (4*intNum),
                                   2UL, /* word register */
                                   1UL ));
      }
      intMask>>=1;
      intNum++;
    }

    /* 2. */
    BX_TRY(BestXDirectRegMaskWrite ( handle,
                     BX_REG_INT_EN_REG,
                     2UL,
                     pci_int,
                     pci_int ));

    /* 3. */
    BX_TRY(BestXDirectRegMaskWrite( handle,
                    BX_REG_INT_ASSERT_REG,
                    sizeof(bx_int16),
                    pci_int,
                    0 ));    

    /* 4. */
    BX_TRY(BestXDirectRegMaskWrite( handle,
                    BX_REG_INT_ASSERT_REG,
                    sizeof(bx_int16),
                    pci_int,
                    pci_int ));    
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype BestXInterruptClear (
 *
 * Purpose      : clear interrupts
 *                this function is not exported, called by BestXStatusClear
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXInterruptClear (
  bx_handletype handle,
  bx_int32 pci_int
  )
{
  BX_TRY_VARS_NO_PROG;
  BX_DECLARE_FUNCNAME("BestXInterruptClear [--]");
  
  BX_TRY_BEGIN {
    bx_int32 clrVal;
    
    BX_FCT_PARAM_CHK(2,pci_int>0xf);

    /* about clearing interrupts:
       1. disable interrupt in mask register ("real" clear can only
          be done via config space
       2. clear interrupt status in DBI shadow register (see note below) */

    /* 1. */
    BX_TRY (BestXDirectRegMaskWrite(handle,
                                    BX_REG_INT_EN_REG,  /* address */
                                    2UL,                /* regwidth */
                                    pci_int,            /* mask */
                                    0UL));              /* value */

    /* NOTE on BX_REG_STAT_IAE_REG:
       This is a DBI shadow register.  If an interrupt is
       currently asserted, this bit is always '1' and will remain
       '1' until this register (independent of
       interrupt-deassertion !) is cleared (i.e. written) again.
       (It will directly be set again, if interrupt is still
       asserted).  The 'real' (i.e. current) status of the
       interrupt signal cannot be read or cleared via DBI but only
       via a PCI config access.  This register is initialized by
       the PU loader only (HW reset value is X), i.e. it may show
       wrong contents, if PU loader has not executed !!  */

    /* Clear shadowed interrupt (a NOP, if interrupt is currently
       asserted) */
       
    /* rather complicated: status interrupt and error register
       has peculiar bit-assignment (it wasn't me... HL) */
    clrVal =
      (pci_int & 1UL << 7) |            /* intA bit0 -> bit7 */
      (pci_int & 2UL << 5) |            /* intB bit1 -> bit6 */
      (pci_int & 4UL << 3) |            /* intC bit2 -> bit5 */
      (pci_int & 8UL << 1);             /* intD bit3 -> bit4 */
    
    BX_TRY(BestXDirectRegWrite( handle,
                                BX_REG_STAT_IAE_REG,
                                2UL,
                                clrVal ));
  }

  return (BX_TRY_RET);
}
/*---------------------------------------------------------------------------*
 * bx_errtype BestXInterruptStatus (
 *
 * Purpose      : status interrupts
 *                this function is not exported, called by BestXStatusRead
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXInterruptStatus (
  bx_handletype handle,
  bx_int32ptr pci_int
  )
{
  BX_TRY_VARS_NO_PROG;
  BX_DECLARE_FUNCNAME("BestXInterruptStatus [--]");
  
  BX_TRY_BEGIN {
    bx_int32 maskVal;
    bx_int32 statVal;
    
    BX_TRY_FCT_PARAM_NULL_POINTER(pci_int);

    /* about interrupt status:
       1. check whether interrupt is masked or not
       2. try to clear all interrupt statuses in iae reg
       3. check whether interrupt status register (still) shows 1 */

    /* 1. */
    BX_TRY (BestXDirectRegRead( handle,
                BX_REG_INT_EN_REG,  /* address */
                2UL,                /* regwidth */
                &maskVal ));

    /* NOTE on BX_REG_STAT_IAE_REG:
       This is a DBI shadow register.  If an interrupt is
       currently asserted, this bit is always '1' and will remain
       '1' until this register (independent of
       interrupt-deassertion !) is cleared (i.e. written) again.
       (It will directly be set again, if interrupt is still
       asserted).  The 'real' (i.e. current) status of the
       interrupt signal cannot be read or cleared via DBI but only
       via a PCI config access.  This register is initialized by
       the PU loader only (HW reset value is X), i.e. it may show
       wrong contents, if PU loader has not executed !!  */

    /* 2. */
    BX_TRY(BestXDirectRegWrite( handle,
                BX_REG_STAT_IAE_REG,
                2UL,
                0xf0UL ));

    /* 3. */
    BX_TRY(BestXDirectRegRead( handle,
                   BX_REG_STAT_IAE_REG,
                   2UL,
                   &statVal ));

    /* rather complicated: status interrupt and error register
       has peculiar bit-assignment (it wasn't me... HL) */
    *pci_int = maskVal &
      ( (statVal & 0x80UL) >> 7 |            /* intA bit7 -> bit0 */
    (statVal & 0x40UL) >> 5 |            /* intB bit6 -> bit1 */
    (statVal & 0x20UL) >> 3 |            /* intC bit5 -> bit2 */
    (statVal & 0x10UL) >> 1 );             /* intD bit4 -> bit3 */
    
  }

  return (BX_TRY_RET);
}

/* -----------------------------------------------------------------
 * LED Functions
 * ----------------------------------------------------------------- */

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXLedSet(
 *
 * Purpose: Switch tiny LEDs to user resp. card-mode.
 * Status:  Not supported function.
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXLedSet(
    bx_handletype handle,
    bx_int32 value
)
{
  BX_DECLARE_FUNCNAME("BestXLedSet [ledset]");

  bx_int8 help = (bx_int8) value;
  bx_errtype err;
  BX_FCT_PARAM_CHK(2, ((value != BX_LED_USER) && (value != BX_LED_CARD)));

  if(BestXHasFirmware(handle))
  {
  err = BestXBasicCommand(handle, CMD_HEX_DISPLAY, &help, IN_HEX_DISPLAY, NULL, NULL);
  }

  BX_ERRETURN(err);
}

/* -----------------------------------------------------------------
 * Hex Display Functions
 * ----------------------------------------------------------------- */
/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXDisplayWrite(
 *
 * Purpose: Write a hex-number to the 4- digit hex-display
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDisplayWrite(
    bx_handletype handle,
    bx_int32 value
)
{
  /* Writes a WORD (16 bit) to board display */

  bx_int16 digit[4]; /* digit[3] is most significant (leftmost) digit */
  bx_int16 d;

  BX_DECLARE_FUNCNAME("BestXDisplayWrite [displaywrite]");

  BX_TRY_VARS_NO_PROG;

  BX_FCT_PARAM_CHK(2,value>0xffff);

  /* We ignore the property BX_BOARD_DISPLAY here, i.e.
     if the display is not in user-mode, the written number
     may be directly overwritten by the card.
  */

  for (d=0;d<4;d++)
  {
    digit[d]=(bx_int16)(value>>d*4&0xf);/* this value is shown on display */
    assert(digit[d]<=0xf);              /* one nibble per digit           */
    digit[d] += (digit[d]>9 ? 'A'-10:'0'); /* corresponding ASCII value   */
  }

  /* Remove leading zeros, except the rightmost digit */
  for (d=0;d<3;d++) /* do not change last digit at index 0 ! */
  {
    if (digit[3-d]=='0')
    {
      /* Replace leading zero by a blank */
      digit[3-d]=' ';
    }
    else
    {
      /* stop the for-loop */
      break;
    }
  }

  /* Now write to display */
  BX_TRY_BEGIN
  {
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_DISP_DIGIT3,sizeof(bx_int16),digit[3]));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_DISP_DIGIT2,sizeof(bx_int16),digit[2]));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_DISP_DIGIT1,sizeof(bx_int16),digit[1]));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_DISP_DIGIT0,sizeof(bx_int16),digit[0]));
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXDisplayStringWrite(
 *
 * Purpose: Write a string to the 4- digit hex-display
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDisplayStringWrite(
    bx_handletype handle,
    bx_charptrtype msg
)
{
  /* Writes a string (4 chars) to board display */

  /* We ignore the property BX_BOARD_DISPLAY here, i.e.
     if the display is not in user-mode, the written string
     may be directly overwritten by the card.
  */

  bx_int16 digit[4]={' ',' ',' ',' '};  /* default string: 4 blanks */
  /* digit[3] is most significant (leftmost) digit */

  bx_int16 d;
  size_t len=0;

  BX_DECLARE_FUNCNAME("BestXDisplayStringWrite [displaystringwrite]");

  BX_TRY_VARS_NO_PROG;

  if (msg)
  {
    len=BESTX_STRLEN(msg);
  }
  else
  {
    /* Passing a NULL pointer is same as
       passing an empty string (blanking).
    */
  }

  if (len>4)
  {
    /* We can only show the first 4 chars */
    len=4;
  }

  for (d=0;d<len;d++)
  {
    if (msg[d]>=0x20 && msg[d]<0x7f)
    {
      /* character is in valid ASCII range for display */
      digit[3-d]=msg[d]; /* overwrite blank */
    }
  }

  /* Write to display */
  BX_TRY_BEGIN
  {
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_DISP_DIGIT3,sizeof(bx_int16),digit[3]));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_DISP_DIGIT2,sizeof(bx_int16),digit[2]));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_DISP_DIGIT1,sizeof(bx_int16),digit[1]));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_DISP_DIGIT0,sizeof(bx_int16),digit[0]));
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXDisplayMessageWrite(
 *
 * Purpose: Write a message to the 4- digit hex-display
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDisplayMessageWrite(
    bx_handletype handle,
    bx_int32 speed,
    bx_charptrtype msg
)
{
  /* Writes a message to board display */

  /* We ignore the property BX_BOARD_DISPLAY here, i.e.
     if the display is not in user-mode, the written string
     may be directly overwritten by the card.
  */
  bx_int32 i,len;
  bx_int32 displayMode=0;
  
  BX_DECLARE_FUNCNAME("BestXDisplayMessageWrite [displaymessagewrite]");

  BX_TRY_VARS_NO_PROG;

  /* Write to display */
  BX_TRY_BEGIN
  {
    if (!BestXIsCore(handle) && bx_handlearray[handle].capable /* capable==NULL during HW update ! */)
    {
      BX_TRY(BestXBoardRead(handle));
      BX_TRY(BestXBoardGet(handle,BX_BOARD_DISPLAY,&displayMode));
      BX_TRY(BestXBoardSet(handle,BX_BOARD_DISPLAY,BX_BOARD_DISPLAY_USER));
      BX_TRY(BestXBoardProg(handle));
    }

    if (msg==NULL) 
    {
      /* Blank display */
      BX_TRY(BestXDisplayStringWrite(handle,NULL));
    }
    else
    {
      len=BESTX_STRLEN(msg);
      for (i=0;i<len;i++)
      {
        if (len-i<8)
        {
          /* Dim intensity down on last characters */  
          BX_TRY(BestXDirectRegWrite(handle,BX_REG_DISP_CTRL0,2,(8-(len-i))<<3));
        }
        /* Write to display and wait some ms */
        BX_TRY(BestXDisplayStringWrite(handle,msg+i));
        BESTX_SLEEP(speed);
      }

      /* Finally clear display */
      BX_TRY(BestXDisplayStringWrite(handle,NULL));

      /* Back to full intensity */
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_DISP_CTRL0,2,0)); 
    }

    if (!BestXIsCore(handle)&& bx_handlearray[handle].capable)
    {
      /* Restore previous display mode */
      BX_TRY(BestXBoardSet(handle,BX_BOARD_DISPLAY,displayMode));
      BX_TRY(BestXBoardProg(handle));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/* -----------------------------------------------------------------
 * Miscellaneous Functions
 * ----------------------------------------------------------------- */

#define MAX_VER_STR_LEN   4096

/*---------------------------------------------------------------------------*
 * static bx_errtype BestXVersionReadCore (
 *
 * Purpose      : ask core for various versions, called by BestXVersionRead
 * Inputs       : handle, versionprop, buffer(!must be allocated!)
 * Outputs      : string pointer
 * Returns      : bx_errtype
 *---------------------------------------------------------------------------*/
static bx_errtype BestXVersionReadCore (
  bx_handletype handle,
  bx_versiontype versionprop,
  bx_charptrtype string
  )
{
  bx_errtype err = BX_E_OK;     /* assume it works... */
  bx_int32 regnum;              /* which version to read? */
  bx_int32 numBytes = 20;       /* length */

  if(!BestXHasFirmware(handle))
  {
    BX_ERRETURN(BX_E_OK);
  }
  
  
  switch (versionprop)
  {
    case BX_VERSION_PRODUCT:
      regnum = RPRODUCT_STRING;
      break;
    case BX_VERSION_SERIALNO:
      regnum = SERIAL_STRING;
      break;
    case BX_VERSION_BOARDID:
      regnum = REVISION_STRING;
      break;
    case BX_VERSION_XILINX:
      regnum = XIL_DATE_CODE;
      numBytes = 4;
      break;
    case BX_VERSION_CORE:
      regnum = CORE_STRING;
      break;

/*
   case BX_VERSION_TEAM:
   case BX_VERSION_MEPHISTO:
   case BX_VERSION_FIRMWARE: */

    default:
      err= BX_E_FCT_PARAM;
  }

  if (err == BX_E_OK)
  {
    err = BestXBasicBlockRead(handle, regnum, (bx_int8 *) string, numBytes);
  }

  return err;
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXVersionRead(
 *
 * Purpose      : Get various version infos from card
 * Inputs       :
 * Outputs      :
 * Returns      :
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXVersionRead(
    bx_handletype handle,
    bx_versiontype versionprop,
    bx_charptrtype * string
)
{
  BX_DECLARE_FUNCNAME("BestXVersionRead [versionread]");

  bx_errtype err=BX_E_OK;
  bx_int32 value=0;
  bx_int32 value2=0;
  static char versionstring[MAX_VER_STR_LEN];
  bx_int16 length = (MAX_VER_STR_LEN - 1);
  bx_int8 in_zw;
  bx_charptrtype pChar=NULL;

  BX_TRY_VARS_NO_PROG;

  BX_FCT_PARAM_NULL_POINTER_CHK(string);

  versionstring[0] = '\0'; /* empty string */

  BX_TRY_BEGIN
  {
    if (!BestX16BitRegisterFile (handle))  /* firmware? */
    {
      BX_TRY(BestXVersionReadCore(handle, versionprop, versionstring));
    }
    else
    {
      bx_hwseriestype hwseries;
      BX_TRY(BestXGetHWSeriesFromHandle(handle,&hwseries));

      switch (versionprop)
      {
        case BX_VERSION_CAPI:
          BESTX_SPRINTF(versionstring,"0x%08lx",BX_VER_NUMBER(xcapi_version(0)));
          break;

        case BX_VERSION_ALTERA:
          if (!BestXHasMephisto(handle))
          {
            /* Altera not available on 30er/23er */
            break;
          }    
        case BX_VERSION_FIRMWARE_DATE:
        case BX_VERSION_XILINX:
        case BX_VERSION_FIRMWARE:
        case BX_VERSION_CORE:
          /* All these are not available for 22/23er ! */
          if (BestXHasFirmware(handle))
          {
            in_zw = (bx_int8) versionprop;
            length = OUT_VERSION_GET;

            err = BestXBasicCommand(handle, CMD_VERSION_GET,
                                    &in_zw, IN_VERSION_GET,
                                    (bx_int8 *) versionstring, &length);
            length = (bx_int16) min(length, (MAX_VER_STR_LEN - 1UL));
            versionstring[length] = '\0';

            /* now convert string, if necessary */
            if(versionprop==BX_VERSION_XILINX||versionprop==BX_VERSION_ALTERA)
            {
              /* convert to human readable date */               
              value = strtoul(versionstring, 0, 16);  /* hexadecimal value */
              BESTX_SPRINTF(versionstring, "%s", ctime((time_t *) & value));
              if (NULL != (pChar = strchr(versionstring, '\n')))
              {
                *pChar = '\0';            /* get rid of '\n' */
              }
            }
          }
          break;

        case BX_VERSION_BOARDID:
          /* Check for valid signature */
          BX_TRY(BestXDirectRegRead(handle,BX_VERSION_SIGNATURE,sizeof(bx_int32),&value));
          if (value==BX_VALID_SIGNATURE)
          {
            BX_TRY(BestXDirectRegBlockRead(handle,BX_BOARD_ID,sizeof(bx_int32),0,
                           (bx_int8ptr)versionstring,sizeof(bx_int8),BX_BOARD_ID_LENGTH));
            versionstring[BX_BOARD_ID_LENGTH-1] = '\0';
            if (versionstring[0]==(char)0xff) 
            {
              BESTX_STRCPY(versionstring,"Not programmed");
            }
          }
          break;
      
        case BX_VERSION_PRODUCT:
          if (BestXHasFirmware(handle))
          {
            in_zw = (bx_int8) versionprop;
            length = OUT_VERSION_GET;

            err = BestXBasicCommand(handle, CMD_VERSION_GET,
                                    &in_zw, IN_VERSION_GET,
                                    (bx_int8 *) versionstring, &length);
            length = (bx_int16) min(length, (MAX_VER_STR_LEN - 1UL));
            versionstring[length] = '\0';
          }
          else
          {
            BX_TRY (BestXDirectRegRead (handle,BX_VERSION_SIGNATURE,4UL,&value));

            if (value == BX_VALID_SIGNATURE) /* flash was programmed */
            {
              /* read product string from flash */
              BX_TRY (BestXDirectRegBlockRead( handle,BX_OVERLOADED_PRODUCT_STRING,4UL,0,(bx_int8ptr) versionstring,1UL,BX_PRODUCT_STRING_LENGTH));
            }
            else
            {
              BESTX_STRCPY(versionstring,"E2922A");
            }
          }
          break;    
    
        case BX_VERSION_SERIALNO:
          /* Check for valid signature */
          BX_TRY(BestXDirectRegRead(handle,BX_VERSION_SIGNATURE,sizeof(bx_int32),&value));
          if (value==BX_VALID_SIGNATURE)
          {
            BX_TRY(BestXDirectRegBlockRead(handle,BX_SERIAL_NUMBER,sizeof(bx_int32),0,
                           (bx_int8ptr)versionstring,sizeof(bx_int8),BX_SERIAL_NUMBER_LENGTH));
            versionstring[BX_SERIAL_NUMBER_LENGTH-1] = '\0';
            if (versionstring[0]==(char)0xff) 
            {
              BESTX_STRCPY(versionstring,"Not programmed");
            }
          }
          break;    

        case BX_VERSION_MEPHISTO:
          if (BestXHasMephisto(handle))
          {
            bx_porttype port;
            BX_TRY(BestXGetPortFromHandle(handle, &port));
            if (port != BX_PORT_OFFLINE)
            {
              BX_TRY(BestXDirectRegRead(handle,BX_REG_CHIP_REV_REG,sizeof(bx_int16),&value));
            }
            else /* offline: simulate version of mephisto to 0x300 */
            {
              value = 0x0300;
            }
            BESTX_SPRINTF(versionstring, "0x%04lx",value);
          }
          else
          {
            /* not available */
            break;
          }
        case BX_VERSION_FAUST: 
          if (BestXHasMephisto(handle))
          {
            /* not available */
            break;
          }
          {
            if (BestXIsOffline(handle))
            {
              /* offline: simulate version of mephisto to 0x310 */
              value = 0x0310;    
            }
            else
            {
              BX_TRY(BestXDirectRegRead(handle,BX_REG_CHIP_REV_REG,sizeof(bx_int16),&value));
              if (value==0x0514)
              {
                /* Check, wether its the first or second cut */
                BX_TRY(BestXDirectRegRead(handle,BX_REG_SYS_RELOAD_EN,sizeof(bx_int16),&value2));
                if ((value2&0x1)==1)
                {
                  /* First cut (keep as 0x0514) */
                }
                else
                {
                  /* Second cut (let's call it 0x0515) */
                  value=0x0515;
                }
              }
            }
            BESTX_SPRINTF(versionstring, "0x%04lx",value);
          }
          break;
    
        case BX_VERSION_COCO: 
          if (!BestXHasFaustFPGA(handle))
          {
            /* not available */
            break;
          }
          else
          {
            if (BestXIsOffline(handle))
            {
                  /* offline: simulate version of COCO to 0x01 */
              value = 0x01;    
            }
            else 
            {
              BX_TRY(BestXDirectRegRead(handle,BX_REG_COCO_VERSION,sizeof(bx_int16),&value));
            }
            BESTX_SPRINTF(versionstring, "0x%02lx",value);
          }
          break;

        case BX_VERSION_TEAM:
          if (BestXHasMephisto(handle))
          {
            BESTX_STRCPY(versionstring,
                "Product Generation: Tilmann Wendel, Achim Loesch\n"
                "Exerciser: Harald Beck\n"
                "Analyzer: Stefan Friebe\n"
                "Processor Board: Harald Beck\n"
                "PPR: Henrik Liebau\n"
                "Driver: Christoph Schmidt\n"
                "Xilinx: Harald Beck\n"
                "Mephisto: Herbert Tiedemann\n"
                "GUI: Christoph Schmidt\n"
                "Firmware: Christoph Schmidt\n"
                "Production: Dieter Klein, Roland Scherzinger");
          }
          else
          {
            BESTX_STRCPY(versionstring,
                "Product Generation: Tilmann Wendel, Thomas Mattes\n"
                "Exerciser: Stefan Friebe\n"
                "Analyzer: Stefan Friebe\n"
                "Processor Board: Friedhelm Hefter\n"
                "PPR: Henrik Liebau\n"
                "Driver: Christoph Schmidt\n"
                "Xilinx: Stefan Friebe\n"
                "Faust: Stefan Friebe\n"
                "GUI: Christoph Schmidt\n"
                "Firmware: Christoph Schmidt\n"
                "Production: Dieter Klein, Roland Scherzinger");
          }

          break;

        default:
          BX_FCT_PARAM_MSG(2, "Invalid property (versionprop)");
          BX_TRY_ERROR(BX_E_FCT_PARAM);
      } /* switch */
    } /* else */

    if (versionstring[0]=='\0')
    {
      /* signature may be missing or 22er */
      BESTX_STRCPY(versionstring,"Not available");
    }
    
    *string = versionstring;
  
  } /* TRY_BEGIN */

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXSMReset(bx_handletype handle)
 *
 * Purpose      :
 * Inputs       :
 * Outputs      :
 * Returns      :
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXSMReset(bx_handletype handle)
{
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    if(!BestXHasFirmware(handle))
  {
      BX_ERRETURN(BX_E_OK);
    }
    BX_TRY(BestXBasicCommand(handle, CMD_SM_RESET,
          NULL, IN_SM_RESET, NULL, NULL));
  }

  BX_ERRETURN(BX_TRY_RET);
}

#define WAIT_RECONNECT (bx_int32)5000     /* wait this long for a reconnect */
#define BOOT_BAUD_E2925  9600

/*---------------------------------------------------------------------------*
 * BestXBoardResetAdv
 *
 * Purpose: Reset board (for biosswitch or something like that)
 *---------------------------------------------------------------------------*/
static bx_errtype BestXBoardResetAdv(bx_handletype handle,bx_int32 boot)
{

  BX_TRY_VARS_NO_PROG;
  BEST_TIMER tmrTotal;
  BX_HANDLECHECK;

  BX_TRY_BEGIN
  {

    if(BestXHasFirmware(handle))
    {
      if (BestX16BitRegisterFile(handle))
      {
        BX_TRY(BestXCoreSwitch(handle));
        BX_TRY(BestXBiosSwitch(handle,1));
      }
      else
      {
        /* Core mode */
        bx_porttype port;
        bx_int32 portnum;
        bx_int32 baud;
        BX_TRY(BestXGetPortFromHandle(handle, &port));
        BX_TRY(BestXGetPortnumberFromHandle(handle, &portnum));
        baud = bx_handlearray[handle].param;

        /* a serial port can only do this at the default baud rate */
        if (port == BX_PORT_RS232 && baud != BOOT_BAUD_E2925)
        {
          BX_TRY(BestXRS232BaudRateSet(handle, BOOT_BAUD_E2925));
        }

        {
          bx_int32 open_tries = RETRY_COUNT_AFTER_BOOT;
          bx_errtype open_err;
          bx_int8 val = (boot?'r':'f'); /* reboot or restart firmware only */

          BX_TRY_FAIL(BestXIsCore(handle) ? BX_E_OK : BX_E_BAD_HANDLE);

          /* reset board, ignore return value, CZ */
          (void) BestXBasicBlockWrite(handle, BOARD_RESET, &val, 1UL);

          /* release connection, so that FHIF doesn't crash */
        
          if (port==BX_PORT_FASTHIF || port==BX_PORT_USB && BestXHasFaust(handle))
          {
            BESTX_SLEEP(1000);
            BestXReleaseConnection(handle);
          }
             
          BestStartTimeout((bx_int32)WAIT_RECONNECT, &tmrTotal);

          BX_E_ERROR_MSG_SET("Timeout.");
          while (BX_E_OK == BestXCheckConnection(handle))
          {
            BX_TRY_FAIL(BestIsTimeoutDone(&tmrTotal) ? BX_E_ERROR : BX_E_OK);
          }

          (void) BestXCloseAdv(handle, BX_CLOSE_RESERVED);

          BESTX_SLEEP(5000); /* 4000 is not enough for PUProg() chris */

          do
          {
            /* Caution: If we have no CAPI capability onboard,
               BestXDBInit() wont get called by BestXOpenAdv() ! */
            open_err = BestXOpenAdv(&handle, port, portnum,
              NULL, BX_OPEN_STANDARD_RESERVED,0,NULL);

          } while ((open_err != BX_E_OK) && --open_tries);

          BX_TRY_FAIL(open_err);
        }

        /* reset baudrate if we changed it */
        if (port == BX_PORT_RS232 && baud != BOOT_BAUD_E2925)
        {
          BX_TRY(BestXRS232BaudRateSet(handle, baud));
        }

        /* When called e.g. via GUI->CLI->puprog->biosswitch  
           and have no CAPI license onboard, we may have to 
           enable it here.
           Problem: No calls into xintern.c allowed here. 
           Possible solution: special implementation in application
           General problem: Saving powerup loader from GUI, when no
           onboard license available, will lead to deletion of DB !
         */

      }
    } /* if(BestXHasFirmware) */
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXBoardReset
 *
 * Purpose:
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXBoardReset(bx_handletype handle)
{
  BX_TRY_VARS_NO_PROG;
  
  BX_DECLARE_FUNCNAME("BestXBoardReset [boardreset]");
  
  BX_TRY_BEGIN
  {
    BX_TRY(BestXBoardResetAdv(handle,1 /* complete reset */));
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXBoardSoftReset
 *
 * Purpose: A BiosSwitch without resetting Mephisto 
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXBoardSoftReset(bx_handletype handle)
{
  BX_TRY_VARS_NO_PROG;
  
  BX_DECLARE_FUNCNAME("BestXBoardSoftReset [boardsoftreset]");
  
  BX_TRY_BEGIN
  {
    BX_TRY(BestXBoardResetAdv(handle,0));
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXBoardRestart(bx_handletype handle)
 *
 * Purpose      :
 * Inputs       :
 * Outputs      :
 * Returns      :
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXBoardRestart(bx_handletype handle)
{
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    if (BestXIsCore(handle))
    {
      BX_TRY(BestXBoardReset(handle));
    }
    else
    {
      if(BestXHasFirmware(handle))
      {
        BX_TRY(BestXBasicCommand(handle, CMD_BOARD_RESTART,
            NULL, IN_BOARD_RESTART, NULL, NULL));
      }
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXCapiPropSet(
 *
 * Purpose      :
 * Inputs       :
 * Outputs      :
 * Returns      :
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXCapiPropSet(
    bx_handletype handle,
    bx_capiproptype capiprop,
    bx_int32 value
)
{
  BX_DECLARE_FUNCNAME("BestXCapiPropSet [capiprpset]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN {
    /* no capability checking */
    BX_TRY_FCT_PARAM (capiprop, capiprop >= BX_CAPIPROP_SIZE);
    bestx_capi_prop [handle] [capiprop] = value;
  }

  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXCapiPropGet(
 *
 * Purpose      :
 * Inputs       :
 * Outputs      :
 * Returns      :
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXCapiPropGet(
    bx_handletype handle,
    bx_capiproptype capiprop,
    bx_int32 *value
)
{
  BX_DECLARE_FUNCNAME("BestXCapiPropGet [capiprpget]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* no capability checking */
    BX_TRY_FCT_PARAM (capiprop, capiprop >= BX_CAPIPROP_SIZE);
    BX_TRY_FCT_PARAM_NULL_POINTER (value);
    *value = bestx_capi_prop [handle] [capiprop];
  }

  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXFlashProg
 *
 * Purpose: Programs the card's Flash EEPROM
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXFlashProg(
  bx_handletype handle,
  bx_int32   FlashAddress,    /* destination flash address */
  bx_int8ptr FlashData,       /* Flashcode */
  bx_int32   NumBytes         /* length of FlashData[] in bytes */
)
{
  BX_DECLARE_FUNCNAME("BestXFlashProg []");

  bx_int8ptr buf=NULL; /* buflen + FlashAddress + FlashData[]                  */ 
  bx_int16 buflen;     /* length of buf[] in bytes without the first two bytes */

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_TRY_FCT_PARAM_NULL_POINTER (FlashData);

    /* buf is as follows:
       - 2 Bytes for the buffer length (BX_LENGTH_NEGOTIATE) = buflen
         buflen does not include the first two bytes !
       - 4 Bytes for the flash startaddress (FlashAddress)
       - FlashData consisting of NumBytes bytes 
    */
      
    buflen=(bx_int16)NumBytes+4; /* 4 bytes for FlashAddress */
    buf=(bx_int8ptr) BestXMemMalloc(sizeof(bx_int8)*(buflen+2)); /* 2 bytes for buflen */

    if (buf==NULL)
    {
      BX_ERRETURN (BX_E_HOST_MEM_FULL);
    }

    /* Put 2 bytes (buflen) for the BX_LENGTH_NEGOTIATE into mystream, bytes 0,1 */
    (void) BestXWord2Stream(buf,&buflen,1UL);
 
    /* Put 4 bytes for the flash startaddress into mystream, bytes 2-5 */
    (void) BestXLong2Stream(buf+2,&FlashAddress,1UL);

    /* Append FlashData[] to buf */ 
    BESTX_MEMCPY(buf+6,FlashData,NumBytes);

    if(BestXHasFirmware(handle))
    {
      BX_TRY(BestXBasicCommand(handle,CMD_FLASH_PROG,buf,BX_LENGTH_NEGOTIATE,NULL,NULL));
    }
  }
  BestXMemFree((void**)&buf);
 
  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype BestXSyncRegWrite
 *
 * Purpose: Writes a value to a register, synced resp. forced
 * Status:  Make number of retries a reserved prop,; insert a sleep ??
 *---------------------------------------------------------------------------*/
bx_errtype BestXSyncRegWrite(
 bx_handletype handle,
 bx_int32 stat_addr,  /* Status register address  */
 bx_int32 stat_size,  /* Status register size     */
 bx_int32 stat_mask,  /* Status register mask     */
 bx_int32 stat_value, /* Status ready value       */
 bx_int32 ctrl_addr,  /* Control register address */
 bx_int32 ctrl_size,  /* Control register size    */
 bx_int32 ctrl_mask,  /* Control register mask    */
 bx_int32 ctrl_value, /* Control register value   */
 bx_int32 sync_addr,  /* Sync register address (currently unused) */
 bx_int32 force_addr, /* Force register address   */
 bx_int32 force_size, /* Force register size      */
 bx_int32 DoSoftForce /* Force it, but try a sync write before */

 /* Not needed:
 bx_int32 sync_size/mask/value  -> equals ctrl_size/mask/value
 */
)
  /* Goal: Write a value (ctrl_value) into a register (ctrl_addr)
           synchronized to the PCI clock or forced.
  
    Here is what we do in the synchronized case (DoHardForce=0):
    1. Check the status register until it gives us its OK.
       This tells us that we are allowed to write to the control register,
       i.e there is no previous syncing pending any more.
       After multiple tries without success, we return an error here.
       (Exception: DoSoftForce=1)
    2. Write to control register (read, modify with mask, write).
    3. Same as 1. 
       After this step we are done, i.e.
       the sync-register equals the control register
       (HW first sets the sync register and then signals ready).
       If waiting was not successful and DoSoftForce=1, we
       force the write instead of returning an error.

    Here is what we do in the forced case (DoHardForce=1):
    1. Write to control register (read, modify with mask, write).
    2. Write an arbitray value to the force register.
       This forces the control-register contents into 
       the sync register immediately. 
    In this case we do no waiting at all for speed purposes !


    Here are the registers, which need syncing:
    1. Decoder enable register
    2. Run control register
    3. Config do retry register
    4. target force retry register
    5. Run/Prog mode registers for clock switching 
      (exerciser,analyzer,performance)
  */
{  
  BX_DECLARE_FUNCNAME("BestXSyncRegWrite []");

  bx_int32 NumTries;   /* chris:TBD: make this a property */

  /* These three decide, wether to force or sync */
  bx_int32 PowerGood;  /* PCI-power present      */
  bx_int32 PciRST;     /* PCI #RST asserted      */
  bx_int32 BusSpeed;   /* BX_BOARD_BUSCLOCK      */

  bx_int32 DoHardForce; /* wether to force directly or (at least try to) sync the write */

  bx_int32 CurVal,NewVal;  /* helpers */

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    if (BestXIsOffline(handle))
    {
      /* Ahh, no hardware to handle. This is easy: */
      BX_ERRETURN(BX_E_OK);
    }
  
    /*
      Synchronized writing will not work, if there is no or only a
      very slow PCI clock present and will lead to error BX_EFW_PCICLOCK_TOO_SLOW.
      On the other hand, forcing a write while PCI clock is present,
      may lead to wrong results and therefore should be avoided.

      We have 3 different cases concerning syncing:
      1. DoHardForce=1, DoSoftForce= X (dont care)
         No syncing done or even tried: Async but fast write. No error is returned.
         Done, if no PCI clock or no PCI power present.
      2. DoHardForce=0, DoSoftForce=0
         Usual (default) case. The write will be synced.
         An error is returned, if syncing not possible (slow or missing PCI clock).
      3. DoHardForce=0, DoSoftForce=1
         A synced write is tried first. If not possible it will be forced. No error is returned.
         This is, when we need the write (dont want an error), but give the HW some time
         to sync. Disadvantage to case 1: Slower.

      In these cases we do a hard forced (immediate,no waiting) write:
      1. No PCI power (i.e. no PCI clock) present.
         This allows programming a card not being
         plugged into a (powered) PCI(X) bus using
         the external power supply.
      [2.] PCI #RST asserted.
           In this case the PCI-clock is ignored automatically,
           so forcing is OK.
      3. User allows forcing by setting a property.
         This means that user guarantees that there is no
         or only a very slow PCI clock.
         Usual scenario: User gets error BX_EFW_PCICLK_TOO_SLOW
         while programming (timeout during sync-write) the card.
    */

    /* This is the third case property */
    BX_TRY(BestXBoardGet(handle,BX_BOARD_BUSCLOCK,&BusSpeed));

    BX_TRY(BestXStatusRead(handle,BX_STAT_PCIRESET,&PciRST));
    BX_TRY(BestXStatusRead(handle,BX_STAT_PCIXPOWER,&PowerGood));

#if 0 /* We allow programming during PCIRST# beeing asserted !! */
    if (PciRST==0) /* #RST is low-active !! */
    {
      /* #RST asserted */
      /* There may be registers, which 
         are beeing cleared during #RST, so programming would make
         no sense.
      */
      /* chris TBD find good error message: 
         "No programming allowed during #RST beeing active" 
      */
      BX_E_ERROR_MSG_SET("BestXSyncRegWrite: No programming allowed during #RST being active");
      BX_TRY_FAIL(BX_E_ERROR);
    }
#endif

    /* We currently do not consider PciRST here, because the state of 
       #RST may have changed between reading the status and programming. 
    */
    DoHardForce = !PowerGood /* || PciRST */ || BusSpeed==BX_BOARD_BUSCLOCK_SLOW; 

    if (!DoHardForce)
    {
      /* Need to (at least try) sync the write.
         We first have to check, wether we are
         allowed to write to register ctrl_addr
         by checking the status register.
      */
      NumTries=3;
      do
      {
        BX_TRY(BestXDirectRegRead(handle,stat_addr,stat_size,&CurVal));
        if ((CurVal & stat_mask) == (stat_value & stat_mask))
        {
          /* Success, we are allowed to write to ctrl_address */
          assert(NumTries);
          break; /* Ends the do-while() loop */
        }
        else
        {
          /* Something is still going on on the card,
             we have to wait and retry.
             Maybe put a BESTX_SLEEP() here.
          */
        }
      } while(--NumTries);

      /* In case we want to force anyway later on, we dont want an error here ! */
      BX_TRY_FAIL((NumTries||DoSoftForce)?BX_E_OK:BX_EFW_PCICLK_TOO_SLOW);
    }

    /* We are allowed to write to ctrl_addr or to force */

    /* Read out current value */
    BX_TRY(BestXDirectRegRead(handle,ctrl_addr,ctrl_size,&CurVal));

    /* Write new value to control register:
       ctrl_mask=1 -> take this bit from ctrl_val (change)
       ctrl_mask=0 -> take this bit from CurVal   (no change)
    */

    NewVal=ctrl_value & ctrl_mask | CurVal & ~ctrl_mask;

    BX_TRY(BestXDirectRegWrite(handle,ctrl_addr,ctrl_size,NewVal));

    if (DoHardForce)
    {
      /* Force previous write command by writing an arbitrary value
         to the force register.
      */
      BX_TRY(BestXDirectRegWrite(handle,force_addr,force_size,0));
    }
    else
    {
      /* We have to wait until the write is synced,
         by checking the status register again.
      */
      NumTries=3;
      do
      {
        BX_TRY(BestXDirectRegRead(handle,stat_addr,stat_size,&CurVal));
        if ((CurVal & stat_mask) == (stat_value & stat_mask))
        {
          /* Success, our write is synced. */
          assert(NumTries);
          break; /* Ends the do-while() loop */
        }
        else
        {
          /* Something is still going on on the card,
             we have to wait and retry.
             Maybe put a BESTX_SLEEP() here.
          */
        }
      } while(--NumTries);
      
      if (NumTries==0 && DoSoftForce)
      {
        /* I really gave the HW some chances to sync,
           but now we force it.
        */
        BX_TRY(BestXDirectRegWrite(handle,force_addr,force_size,0));
      }

      /* The following may happen here, when called from ExerciserRun/ProgMode():
         NumTries==0 and SoftForce==0
         => this function returns an error (PCICLK_TOO_SLOW)
         => variable handle->CardIsInProgMode is not adapted in ExerciserRun/ProgMode()
         If clock occurs afterwards, HW switches to Run/ProgMode but SW gets no notification.
         => variable handle->CardIsInProgMode has wrong contents
         Possible solution: 
         1. Restore old value by (sync) writing back the previously read value (may cause other problems).
         2. User should set property BX_BOARD_BUSCLOCK 
         3. Check (and adapt) variable handle->CardIsInProgMode at important places, e.g. ExerciserRun() etc. 
       */
      
      BX_TRY_FAIL(NumTries||DoSoftForce?BX_E_OK:BX_EFW_PCICLK_TOO_SLOW);
    }
  }

  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_int32 EXPORT BestXVersionCompare(
 *
 * Purpose      :
 *---------------------------------------------------------------------------*/
bx_int32 EXPORT BestXVersionCompare(
  const bx_vertype *ver1,
  const bx_vertype *ver2
)
{
  /* Compares the given two versions.
     Returns: 0 - if equal,
              1 - if only patchlevel differs,
             >1 - else
  */

  bx_int32 ret=0; /* return value */

  assert(ver1);
  assert(ver2);

  if (ver1==NULL || ver2==NULL)
  {
    /* We should never get here;
       valid pointers guaranteed by caller */
    return 0xffffffff;
  }

  if (ver1->PatchVersion!=ver2->PatchVersion)
  {
    ret |= 1;
  }

  if (ver1->LevelVersion!=ver2->LevelVersion)
  {
    ret |= 2;
  }

  if (ver1->MinorVersion!=ver2->MinorVersion)
  {
    ret |= 4;
  }

  if (ver1->MajorVersion!=ver2->MajorVersion)
  {
    ret |= 8;
  }

  return ret;
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT EraseFlash ( bx_handletype handle,
 *
 * Purpose      : 
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXEraseFlash ( bx_handletype handle,
                                    bx_int32 addrNum,
                                    const bx_int32ptr pAddr )
{
  BX_TRY_VARS_NO_PROG;
  size_t i;
  bx_int32 val32;
  int tryCount;
  
  DBG_OUT(("Erasing flash sectors...\n"));

  BX_TRY_BEGIN {
    
    /* sector erase (8*8K) */
    for (i=0; i < addrNum; i++)
    {
      /* prepare */
      BX_TRY (BestXDirectRegWrite (handle, 0x555<<1, 2UL, 0xaa));
      BX_TRY (BestXDirectRegWrite (handle, 0x2aa<<1, 2UL, 0x55));
      BX_TRY (BestXDirectRegWrite (handle, 0x555<<1, 2UL, 0x80));
      BX_TRY (BestXDirectRegWrite (handle, 0x555<<1, 2UL, 0xaa));
      BX_TRY (BestXDirectRegWrite (handle, 0x2aa<<1, 2UL, 0x55));


      /* printf ("\rErasing sector at addr %04x\n", pAddr[i]); */
      BX_TRY (BestXDirectRegWrite (handle, pAddr[i], 2UL, 0x30));

      tryCount = 0;
      do                                /* check if erasing complete */
      {
        BX_TRY (BestXDirectRegRead(handle, pAddr[i], 2UL, &val32));
        if (((val32 >> 8) & (1<<7)) != 0 )
        {
          break;
        }
        tryCount++;
      }
      while ( tryCount < 0x10000 );

      if (tryCount == 0x10000)
      {
        printf ("\n");
        fprintf (stderr, "\nTimeout at addr %06lx: %lx\n",
                 pAddr[i], val32);
      }
    }
  }
  DBG_OUT(("Erasing done\n"));

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXWriteFlash ( bx_handletype handle,
 *
 * Purpose      : 
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXWriteFlash(bx_handletype handle,
                                  bx_int32 startAddr,
                                  bx_int8 * LoaderCode,
                                  bx_int32 LoaderLength)
{
  BX_TRY_VARS;
  bx_int32 val32;
  bx_int8 dataBuf[2];
  bx_int32 addr;
  int tryCount;
   
  BX_TRY_BEGIN 
  {
    addr = startAddr;
    assert(!(LoaderLength%2));
    
    while (addr< (startAddr+LoaderLength) )
    {
      /* read */
      dataBuf[0]=*LoaderCode;
      dataBuf[1]=*(LoaderCode+1);
      LoaderCode+=2;
      
      /* write data into flash */
      val32 = (((bx_int32)dataBuf[0] << 8) | (bx_int32)dataBuf[1]) &
        0x0000FFFFUL;

      BX_TRY (BestXDirectRegWrite (handle, 0x555<<1, 2UL, 0xaaaa));
      BX_TRY (BestXDirectRegWrite (handle, 0x2aa<<1, 2UL, 0x5555));
      BX_TRY (BestXDirectRegWrite (handle, 0x555<<1, 2UL, 0xa0a0));
      /* now: write ! */
      /* printf("\rWriting value 0x%lx at address 0x%lx\n",val32,addr); */
      BX_TRY (BestXDirectRegWrite (handle, addr,     2UL, val32 ));

      tryCount = 0;
      do
      {
        BX_TRY (BestXDirectRegRead (handle, addr, 2UL, &val32));
        tryCount++;
      }
      while ( tryCount < 100000 && (((bx_int8)(val32>>8)) ^ dataBuf[0]) & (1<<7) );

      if (tryCount == 100000)
      {
        printf ("\n");
        fprintf (stderr, "\nTimeout at addr %06lx: %lx %x\n",
                 addr, val32, dataBuf[0]);
        BX_TRY_FAIL (BX_E_ERROR);
      }
  
      addr+=2;
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}
